#include <stdio.h>

typedef int boolean;
#define true	1
#define false	0

/* ------------------------------------------------------------------ */
/* | Copyright Unpublished, MIPS Computer Systems, Inc.  All Rights | */
/* | Reserved.  This software contains proprietary and confidential | */
/* | information of MIPS and its suppliers.  Use, disclosure or     | */
/* | reproduction is prohibited without the prior express written   | */
/* | consent of MIPS.                                               | */
/* ------------------------------------------------------------------ */
union mips_instruction {
        unsigned word;
        unsigned char byte[4];
        struct {
                unsigned opcode : 6;
                unsigned target : 26;
        } j_format;
        struct {
                unsigned opcode : 6;
                unsigned rs : 5;
                unsigned rt : 5;
                int simmediate : 16;
        } i_format;
        struct {
                unsigned opcode : 6;
                unsigned rs : 5;
                unsigned rt : 5;
                unsigned uimmediate : 16;
        } u_format;
        struct {
                unsigned opcode : 6;
                unsigned rs : 5;
                unsigned rt : 5;
                unsigned rd : 5;
                unsigned re : 5;
                unsigned func : 6;
        } r_format;
        struct {
                unsigned opcode : 6;
                unsigned : 1;
                unsigned fmt : 4;
                unsigned rt : 5;
                unsigned rd : 5;
                unsigned re : 5;
                unsigned func : 6;
        } f_format;
};

#define spec_op         0x00
#define bcond_op        0x01
#define j_op            0x02
#define jal_op          0x03

#define beq_op          0x04
#define bne_op          0x05
#define blez_op         0x06
#define bgtz_op         0x07

#define addi_op         0x08
#define addiu_op        0x09
#define slti_op         0x0A
#define sltiu_op        0x0B

#define andi_op         0x0C
#define ori_op          0x0D
#define xori_op         0x0E
#define lui_op          0x0F

#define lb_op           0x20
#define lh_op           0x21
#define lw_op           0x23
#define lbu_op          0x24
#define lhu_op          0x25
#define ld_op           0x27
#define sb_op           0x28
#define sh_op           0x29
#define sw_op           0x2B
#define sd_op           0x2F
#define lwl_op          0x22
#define lwr_op          0x26
#define swl_op          0x2a
#define swr_op          0x2e

/* Co-processor sub-opcodes */
#define bc_op           0x08
#define mfc_op          0x00
#define cfc_op          0x02
#define mtc_op          0x04
#define ctc_op          0x06

/* Co-processor 0 opcodes */
#define cop0_op         0x10
#define lwc0_op         0x30
#define ldc0_op         0x34
#define swc0_op         0x38
#define sdc0_op         0x3c

/* Co-processor 0 sub-opcodes */
#define tlbr_op         0x1
#define tlbwi_op        0x2
#define tlbwr_op        0x6
#define tlbp_op         0x8
#define rfe_op          0x10
 
/* Co-processor 1 opcodes */
#define cop1_op         0x11
#define lwc1_op         0x31
#define ldc1_op         0x35
#define swc1_op         0x39
#define sdc1_op         0x3D

/* Co-processor 1 sub-opcodes */
#define fadd_op         0x00
#define fsub_op         0x01
#define fmpy_op         0x02
#define fdiv_op         0x03
#define fsqrt_op        0x04
#define fabs_op         0x05
#define fmov_op         0x06
#define fneg_op         0x07
#define fcvts_op        0x20
#define fcvtd_op        0x21
#define fcvte_op        0x22
#define fcvtw_op        0x24
#define fcmp_op         0x30
#define s_fmt           0
#define d_fmt           1
#define e_fmt           2
#define w_fmt           4

/* Other coprocessor opcodes */
#define cop2_op         0x12
#define lwc2_op         0x32
#define ldc2_op         0x36
#define swc2_op         0x3a
#define sdc2_op         0x3e

#define cop3_op         0x13
#define lwc3_op         0x33
#define ldc3_op         0x37
#define swc3_op         0x3b
#define sdc3_op         0x3f


/* bcond subopcodes */
#define bltz_op         0x00
#define bgez_op         0x01
#define bltzal_op       0x10
#define bgezal_op       0x11

/* special subopcodes */
#define sll_op          0x00
#define srl_op          0x02
#define sra_op          0x03
#define sllv_op         0x04
#define srlv_op         0x06
#define srav_op         0x07
#define jr_op           0x08
#define jalr_op         0x09
#define syscall_op      0x0C
#define break_op        0x0D
#define vcall_op        0x0E

#define mfhi_op         0x10
#define mthi_op         0x11
#define mflo_op         0x12
#define mtlo_op         0x13
#define mult_op         0x18
#define multu_op        0x19
#define div_op          0x1A
#define divu_op         0x1B

#define add_op          0x20
#define addu_op         0x21
#define and_op          0x24
#define or_op           0x25
#define xor_op          0x26
#define nor_op          0x27
#define sub_op          0x22
#define subu_op         0x23
#define slt_op          0x2A
#define sltu_op         0x2B

/* register definitions */
#define ZERO 0

static char *op_name[64] = {
/* 0 */	"spec",	"bcond","j",	"jal",	"beq",	"bne",	"blez",	"bgtz",
/* 8 */	"addi",	"addiu","slti",	"sltiu","andi",	"ori",	"xori",	"lui",
/*16 */	"c0",	"c1",	"c2",	"c3",	"op50",	"op54",	"op58",	"op5c",
/*24 */	"op60",	"op64",	"op68",	"op6c",	"op70",	"op74",	"op78",	"op7c",
/*32 */	"lb",	"lh",	"lwl",	"lw",	"lbu",	"lhu",	"lwr",	"ld",
/*40 */	"sb",	"sh",	"swl",	"sw",	"opb0",	"opb4",	"swr",	"sd",
/*48 */	"lwc0",	"lwc1",	"lwc2",	"lwc3",	"ldc0",	"ldc1",	"ldc2",	"ldc3",
/*56 */	"swc0",	"swc1",	"swc2",	"swc3",	"sdc0",	"sdc1",	"sdc2",	"sdc3"
};

static char *spec_name[64] = {
/* 0 */	"sll",	"spec01","srl",	"sra",	"sllv",	"spec05","srlv","srav",
/* 8 */	"jr",	"jalr",	"spec12","spec13","syscall","break","vcall","spec17",
/*16 */	"mfhi",	"mthi",	"mflo",	"mtlo",	"spec24","spec25","spec26","spec27",
/*24 */	"mult",	"multu","div",	"divu",	"spec34","spec35","spec36","spec37",
/*32 */	"add",	"addu",	"sub",	"subu",	"and",	"or",	"xor",	"nor",
/*40 */	"spec50","spec51","slt","sltu",	"spec54","spec55","spec56","spec57",
/*48 */	"spec60","spec61","spec62","spec63","spec64","spec65","spec66","spec67",
/*56 */	"spec70","spec71","spec72","spec73","spec74","spec75","spec76","spec77"
};

static char *bcond_name[32] = {
	"bltz",
	"bgez",
	"bcond02",
	"bcond03",
	"bcond04",
	"bcond05",
	"bcond06",
	"bcond07",
	"bcond08",
	"bcond09",
	"bcond0a",
	"bcond0b",
	"bcond0c",
	"bcond0d",
	"bcond0e",
	"bcond0f",
	"bltzal",
	"bgezal",
	"bcond12",
	"bcond13",
	"bcond14",
	"bcond15",
	"bcond16",
	"bcond17",
	"bcond18",
	"bcond19",
	"bcond1a",
	"bcond1b",
	"bcond1c",
	"bcond1d",
	"bcond1e",
	"bcond1f"
};

static char *cop1_name[64] = {
/* 0 */	"add",	"sub",	"mul",	"div",	"sqrt", "abs",	"mov",	"neg",
/* 8 */	"fop08","fop09","fop0a","fop0b","fop0c","fop0d","fop0e","fop0f",
/*16 */	"fop10","fop11","fop12","fop13","fop14","fop15","fop16","fop17",
/*24 */	"fop18","fop19","fop1a","fop1b","fop1c","fop1d","fop1e","fop1f",
/*32 */	"cvt.s","cvt.d","cvt.e","fop23","cvt.w","fop25","fop26","fop27",
/*40 */	"fop28","fop29","fop2a","fop2b","fop2c","fop2d","fop2e","fop2f",
/*48 */	"c.f", "c.un","c.eq","c.ueq","c.olt","c.ult","c.ole","c.ule",
/*56 */	"c.sf","c.ngle","c.seq","c.ngl","c.lt","c.nge","c.le","c.ngt"
};

static char *fmt_name[16] = {
	"s",	"d",	"e",	"q",
	"w",	"fmt5",	"fmt6",	"fmt7",
	"fmt8",	"fmt9",	"fmta",	"fmtb",
	"fmtc",	"fmtd",	"fmte",	"fmtf"
};

/* public */
/* Three sets of commonly used register names */
/* const */ char *dis_reg_names[3][33] = {
	{	/* compiler names */
		"zero",	"at",	"v0",	"v1",	"a0",	"a1",	"a2",	"a3",
		"t0",	"t1",	"t2",	"t3",	"t4",	"t5",	"t6",	"t7",
		"s0",	"s1",	"s2",	"s3",	"s4",	"s5",	"s6",	"s7",
		"t8",	"t9",	"k0",	"k1",	"gp",	"sp",	"s8",	"ra",
		"pc"
	},
	{	/* hardware names */
		"r0",	"r1",	"r2",	"r3",	"r4",	"r5",	"r6",	"r7",
		"r8",	"r9",	"r10",	"r11",	"r12",	"r13",	"r14",	"r15",
		"r16",	"r17",	"r18",	"r19",	"r20",	"r21",	"r22",	"r23",
		"r24",	"r25",	"r26",	"r27",	"gp",	"sp",	"r30",	"r31",
		""
	},
	{	/* assembler names */
		"$0",	"$at",	"$2",	"$3",	"$4",	"$5",	"$6",	"$7",
		"$8",	"$9",	"$10",	"$11",	"$12",	"$13",	"$14",	"$15",
		"$16",	"$17",	"$18",	"$19",	"$20",	"$21",	"$22",	"$23",
		"$24",	"$25",	"$26",	"$27",	"$gp",	"$sp",	"$30",	"$31",
		""
	}
};

static char *c0_opname[32] = {
	"c0op0","tlbr",	"tlbwi","c0op3","c0op4","c0op5","tlbwr","c0op7",
	"tlbp",	"c0op9","c0op10","c0op11","c0op12","c0op13","c0op14","c0op15",
	"rfe",	"c0op17","c0op18","c0op19","c0op20","c0op21","c0op22","c0op23",
	"c0op24","c0op25","c0op26","c0op27","c0op28","c0op29","c0op30","c0op31"
};

static char *c0_reg[32] = {
	"index","random","tlblo","c0r3","context","c0r5","c0r6","c0r7",
	"badvaddr","c0r9","tlbhi","c0r11","sr",	"cause","epc",	"c0r15",
	"c0r16","c0r17","c0r18","c0r19","c0r20","c0r21","c0r22","c0r23",
	"c0r24","c0r25","c0r26","c0r27","c0r28","c0r29","c0r30","c0r31"
};

/* Remember the options set by dis_init */
#define HARDWARE_NAMES dis_reg_names[1]
#define COMPILER_NAMES dis_reg_names[0]
#define NAME_DEFAULT COMPILER_NAMES
struct {
  char **reg_names;
  int print_jal_targets;
  } save = {
    NAME_DEFAULT,
    true
    };

/* Update regmask to reflect the use of this general-purpose (not fp)
  register, and return its name */
static char *
register_name(ireg, regmask)
  unsigned ireg, *regmask;
  {
  *regmask |= (1 << ireg);
  return save.reg_names[ireg];
  }

/* public -- see .h file */
int
disasm(buffer, address, iword, regmask, symbol_value, ls_register)
  char *buffer;
  unsigned address, iword, *regmask, *symbol_value, *ls_register;
{
	int return_value = 0;
	char *bufptr = buffer;
	boolean do_b_displacement = false;
	boolean do_loadstore = false;
	union mips_instruction i;

	i.word = iword;
	*regmask = *symbol_value = *ls_register = 0;

	switch (i.j_format.opcode) {

	case spec_op:

		if (i.word == 0) {
			strcat(bufptr, "nop");
			bufptr += strlen(bufptr);
			break;
		}
		else if (i.r_format.func == addu_op && i.r_format.rt == ZERO) {
			sprintf(bufptr, "move %s,%s",
			    register_name(i.r_format.rd, regmask),
			    register_name(i.r_format.rs, regmask));
			bufptr += strlen(bufptr);
			break;
		}
		strcat(bufptr, spec_name[i.r_format.func]);
		bufptr += strlen(bufptr);

		switch (i.r_format.func) {
		case sll_op:
		case srl_op:
		case sra_op:
			sprintf(bufptr, " %s,%s,%d",
			    register_name(i.r_format.rd, regmask),
			    register_name(i.r_format.rt, regmask),
			    i.r_format.re);
			break;
		case sllv_op:
		case srlv_op:
		case srav_op:
			sprintf(bufptr, " %s,%s,%s",
			    register_name(i.r_format.rd, regmask),
			    register_name(i.r_format.rt, regmask),
			    register_name(i.r_format.rs, regmask));
			break;
		case mfhi_op:
		case mflo_op:
			sprintf(bufptr, " %s",
			  register_name(i.r_format.rd, regmask));
			break;
		case jr_op:
		case jalr_op:
			return_value = 2;
			/* fall through */
		case mtlo_op:
		case mthi_op:
			sprintf(bufptr, " %s",
			  register_name(i.r_format.rs, regmask));
			break;
		case mult_op:
		case multu_op:
		case div_op:
		case divu_op:
			sprintf(bufptr, " %s,%s",
			    register_name(i.r_format.rs, regmask),
			    register_name(i.r_format.rt, regmask));
			break;
		case syscall_op:
			break;
		case break_op:
		case vcall_op:
			sprintf(bufptr, " %d", i.r_format.rs*32+i.r_format.rt);
			break;
		default:
			sprintf(bufptr, " %s,%s,%s",
			    register_name(i.r_format.rd, regmask),
			    register_name(i.r_format.rs, regmask),
			    register_name(i.r_format.rt, regmask));
			break;
		};
		break;

	case bcond_op:
		sprintf(bufptr, "%s%s %s,",
		    bcond_name[i.i_format.rt & 1],
		    ((i.i_format.rt&0x10) != 0) ? "al" : "",
		    register_name(i.i_format.rs, regmask));
		do_b_displacement = true;
		break;
	case blez_op:
	case bgtz_op:
		sprintf(bufptr, "%s %s,", op_name[i.i_format.opcode],
		    register_name(i.i_format.rs, regmask));
		do_b_displacement = true;
		break;
	case beq_op:
		if (i.i_format.rs == ZERO && i.i_format.rt == ZERO) {
			strcat(bufptr, "b ");
			do_b_displacement = true;
			break;
		}
		/* fall through */
	case bne_op:
		sprintf(bufptr, "%s %s,%s,", op_name[i.i_format.opcode],
		    register_name(i.i_format.rs, regmask),
		    register_name(i.i_format.rt, regmask));
		do_b_displacement = true;
		break;
	case cop0_op:
		switch (i.r_format.rs) {
		case bc_op:
			sprintf(bufptr, "bc0%c ", "ft"[i.r_format.rt]);
			do_b_displacement = true;
			break;
		case mtc_op:
			sprintf(bufptr, "mtc0 %s,%s",
			    register_name(i.r_format.rt, regmask),
			    c0_reg[i.f_format.rd]);
			break;
		case mfc_op:
			sprintf(bufptr, "mfc0 %s,%s",
			    register_name(i.r_format.rt, regmask),
			    c0_reg[i.f_format.rd]);
			break;
		default:
			sprintf(bufptr, "c0 %s", c0_opname[i.f_format.func]);
			break;
		};
		break;
	case cop1_op:
		switch (i.r_format.rs) {
		case bc_op:
			sprintf(bufptr, "bc1%c ", "ft"[i.r_format.rt]);
			do_b_displacement = true;
			break;
		case mtc_op:
			sprintf(bufptr, "mtc1 %s,f%d",
			    register_name(i.r_format.rt, regmask),
			    i.f_format.rd);
			break;
		case mfc_op:
			sprintf(bufptr, "mfc1 %s,f%d",
			    register_name(i.r_format.rt, regmask),
			    i.f_format.rd);
			break;
		case cfc_op:
			sprintf(bufptr, "cfc1 %s,f%d",
			    register_name(i.r_format.rt, regmask),
			    i.f_format.rd);
			break;
		case ctc_op:
			sprintf(bufptr, "ctc1 %s,f%d",
			    register_name(i.r_format.rt, regmask),
			    i.f_format.rd);
			break;
		default:
			sprintf(bufptr, "%s.%s ",
			    cop1_name[i.f_format.func],
			    fmt_name[i.f_format.fmt]);
			bufptr += strlen(bufptr);
			switch (i.f_format.func) {
			  case fsqrt_op:
			  case fabs_op:
			  case fmov_op:
			  case fcvts_op:
			  case fcvtd_op:
			  case fcvte_op:
			  case fcvtw_op:
			    sprintf(bufptr, "f%d,f%d",
				   i.f_format.re,
				   i.f_format.rd);
			    break;
			  case fcmp_op+0x0:
			  case fcmp_op+0x1:
			  case fcmp_op+0x2:
			  case fcmp_op+0x3:
			  case fcmp_op+0x4:
			  case fcmp_op+0x5:
			  case fcmp_op+0x6:
			  case fcmp_op+0x7:
			  case fcmp_op+0x8:
			  case fcmp_op+0x9:
			  case fcmp_op+0xa:
			  case fcmp_op+0xb:
			  case fcmp_op+0xc:
			  case fcmp_op+0xd:
			  case fcmp_op+0xe:
			  case fcmp_op+0xf:
			    sprintf(bufptr, "f%d,f%d",
				   i.f_format.rd,
				   i.f_format.rt);
			    break;
			  default:
			    sprintf(bufptr, "f%d,f%d,f%d",
				   i.f_format.re,
				   i.f_format.rd,
				   i.f_format.rt);
			    break;
			}
			break;
		};
		break;

	case jal_op:
		sprintf(bufptr, "%s ", op_name[i.j_format.opcode]);
		*symbol_value =
		  ((address+4)&~((1<<28)-1)) + (i.j_format.target<<2);
		if (save.print_jal_targets)
		  {
		  bufptr += strlen(bufptr);
		  sprintf(bufptr, "%#x", *symbol_value);
		  }
		return_value = 1;
		break;
		
	case j_op:
		sprintf(bufptr, "%s %#x", op_name[i.j_format.opcode],
		  ((address+4)&~((1<<28)-1)) +(i.j_format.target<<2));
		return_value = 2;
		break;

	case swc1_op:
	case sdc1_op:
	case lwc1_op:
	case ldc1_op:
		sprintf(bufptr, "%s f%d,", op_name[i.i_format.opcode],
		    i.i_format.rt);
		do_loadstore = true;
		break;

	case lb_op:
	case lh_op:
	case lw_op:
	case ld_op:
	case lbu_op:
	case lhu_op:
	case sb_op:
	case sh_op:
	case sw_op:
	case sd_op:
	case lwl_op:
	case lwr_op:
		sprintf(bufptr, "%s %s,", op_name[i.i_format.opcode],
		    register_name(i.i_format.rt, regmask));
		do_loadstore = true;
		break;

	case ori_op:
	case xori_op:
		if (i.u_format.rs == ZERO) {
			sprintf(bufptr, "li %s,%d",
			    register_name(i.u_format.rt, regmask),
			    i.u_format.uimmediate);
			break;
		}
		/* fall through */
	case andi_op:
		sprintf(bufptr, "%s %s,%s,%#x", op_name[i.u_format.opcode],
		    register_name(i.u_format.rt, regmask),
		    register_name(i.u_format.rs, regmask),
		    i.u_format.uimmediate);
		break;
	case lui_op:
		sprintf(bufptr, "%s %s,%#x", op_name[i.u_format.opcode],
		    register_name(i.u_format.rt, regmask),
		    i.u_format.uimmediate);
		break;
	case addi_op:
	case addiu_op:
		if (i.i_format.rs == ZERO) {
			short sign_extender = i.i_format.simmediate;
			sprintf(bufptr, "li %s,%d",
			    register_name(i.i_format.rt, regmask),
			    sign_extender);
			break;
		}
		/* fall through */
	default:
		{
		short sign_extender = i.i_format.simmediate;
		sprintf(bufptr, "%s %s,%s,%d", op_name[i.i_format.opcode],
		    register_name(i.i_format.rt, regmask),
		    register_name(i.i_format.rs, regmask),
		    sign_extender);
		}
		break;
	}

	/* Some instructions require more than just registers */

	if (do_loadstore)
	  {
		short sign_extender = i.i_format.simmediate;
		*symbol_value = sign_extender;
		*ls_register = i.i_format.rs;
		bufptr += strlen(bufptr);
		sprintf(bufptr, "%d(%s)", sign_extender,
		  register_name(i.i_format.rs, regmask));
		return_value = -1;
	  }
	else if (do_b_displacement)
	  {
		short sign_extender = i.i_format.simmediate;
		bufptr += strlen(bufptr);
		sprintf(bufptr, "%#x", address+4+(sign_extender<<2));
		return_value = 2;
	  }

	return return_value;
}
